import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Sparkles, Star, Calendar, Building2, Award, Heart, TrendingUp, Clock } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyVisits.css'

const MyVisits = () => {
  const [visits, setVisits] = useState([])

  useEffect(() => {
    loadVisits()
    const interval = setInterval(loadVisits, 500)
    return () => clearInterval(interval)
  }, [])

  const loadVisits = () => {
    const saved = localStorage.getItem('spaVisits')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setVisits(parsed)
      } catch (e) {
        setVisits([])
      }
    } else {
      const defaultVisits = [
        {
          id: 1,
          procedure: 'Массаж',
          salon: 'SpaLux Premium',
          date: '2025-03-15',
          rating: 5
        },
        {
          id: 2,
          procedure: 'Пилинг',
          salon: 'SpaLux Center',
          date: '2025-03-10',
          rating: 4
        },
        {
          id: 3,
          procedure: 'Обёртывание',
          salon: 'SpaLux Premium',
          date: '2025-03-05',
          rating: 5
        }
      ]
      setVisits(defaultVisits)
      localStorage.setItem('spaVisits', JSON.stringify(defaultVisits))
    }
  }

  const getProcedureColor = (procedure) => {
    const colors = {
      'Массаж': '#D4A574',
      'Пилинг': '#C9A961',
      'Обёртывание': '#E8C5A0',
      'Ароматерапия': '#B8956A'
    }
    return colors[procedure] || '#D4A574'
  }

  const renderStars = (rating) => {
    return Array.from({ length: 5 }, (_, i) => (
      <Star
        key={i}
        size={20}
        fill={i < rating ? '#F59E0B' : 'none'}
        stroke={i < rating ? '#F59E0B' : '#D1D5DB'}
        strokeWidth={2}
      />
    ))
  }

  const totalVisits = visits.length
  const averageRating = visits.length > 0 
    ? (visits.reduce((sum, v) => sum + v.rating, 0) / visits.length).toFixed(1)
    : 0
  const uniqueSalons = new Set(visits.map(v => v.salon)).size

  return (
    <div className="my-visits">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 30 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6 }}
        >
          <motion.div 
            className="hero-icon-wrapper"
            initial={{ scale: 0, rotate: -180 }}
            animate={{ scale: 1, rotate: 0 }}
            transition={{ type: "spring", stiffness: 200, damping: 15, delay: 0.3 }}
          >
            <Sparkles size={96} strokeWidth={2.5} className="hero-icon" />
          </motion.div>
          <h1 className="hero-title">Мои посещения</h1>
          <p className="hero-description">
            История ваших спа-процедур и релаксационных сеансов
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.1, rotate: 4 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(212, 165, 116, 0.2)' }}>
              <Heart size={44} strokeWidth={2.5} color="#D4A574" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalVisits}</div>
              <div className="stat-label">Посещений</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.1, rotate: -4 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(245, 158, 11, 0.2)' }}>
              <Star size={44} strokeWidth={2.5} color="#F59E0B" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ scale: 1.1, rotate: 4 }}
            transition={{ type: "spring", stiffness: 300, damping: 10 }}
          >
            <div className="stat-icon-wrapper" style={{ background: 'rgba(201, 169, 97, 0.2)' }}>
              <Building2 size={44} strokeWidth={2.5} color="#C9A961" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{uniqueSalons}</div>
              <div className="stat-label">Салона</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/spa-1.jpg',
          '/images/spa-2.jpg',
          '/images/spa-3.jpg',
          '/images/spa-4.jpg'
        ]} />

        <motion.section 
          className="benefits-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.6, delay: 0.4 }}
        >
          <div className="benefits-card">
            <h2 className="benefits-title">Почему выбирают нас</h2>
            <div className="benefits-grid">
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(212, 165, 116, 0.2)' }}>
                  <Award size={40} strokeWidth={2.5} color="#D4A574" />
                </div>
                <h3 className="benefit-title">Премиум-качество</h3>
                <p className="benefit-text">Только лучшие процедуры и средства</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(201, 169, 97, 0.2)' }}>
                  <Heart size={40} strokeWidth={2.5} color="#C9A961" />
                </div>
                <h3 className="benefit-title">Забота о здоровье</h3>
                <p className="benefit-text">Натуральные компоненты и методики</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(232, 197, 160, 0.2)' }}>
                  <Clock size={40} strokeWidth={2.5} color="#E8C5A0" />
                </div>
                <h3 className="benefit-title">Гибкий график</h3>
                <p className="benefit-text">Запись в удобное для вас время</p>
              </div>
              <div className="benefit-item">
                <div className="benefit-icon" style={{ background: 'rgba(245, 158, 11, 0.2)' }}>
                  <TrendingUp size={40} strokeWidth={2.5} color="#F59E0B" />
                </div>
                <h3 className="benefit-title">Высокий рейтинг</h3>
                <p className="benefit-text">99% довольных клиентов</p>
              </div>
            </div>
          </div>
        </motion.section>

        <motion.section 
          className="visits-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.6, delay: 0.6 }}
        >
          <h2 className="section-title">Ваши процедуры</h2>
          {visits.length > 0 ? (
            <div className="visits-grid">
              {visits.map((visit, index) => (
                <motion.div 
                  key={visit.id} 
                  className="visit-card"
                  initial={{ opacity: 0, scale: 0.9, rotate: -5 }}
                  animate={{ opacity: 1, scale: 1, rotate: 0 }}
                  transition={{ delay: index * 0.15, duration: 0.5 }}
                  whileHover={{ 
                    scale: 1.1, 
                    rotate: 5,
                    boxShadow: `0 28px 56px ${getProcedureColor(visit.procedure)}40`
                  }}
                  style={{ 
                    borderLeft: `6px solid ${getProcedureColor(visit.procedure)}`,
                    background: `linear-gradient(135deg, rgba(255, 255, 255, 0.99) 0%, rgba(255, 248, 240, 0.98) 100%)`
                  }}
                >
                  <div className="visit-header">
                    <div className="visit-procedure-badge" style={{ background: getProcedureColor(visit.procedure) + '25', color: getProcedureColor(visit.procedure) }}>
                      {visit.procedure}
                    </div>
                    <div className="visit-rating">
                      {renderStars(visit.rating)}
                    </div>
                  </div>
                  <div className="visit-content">
                    <div className="visit-info-item">
                      <Building2 size={24} strokeWidth={2.5} color={getProcedureColor(visit.procedure)} />
                      <span className="visit-info-text">{visit.salon}</span>
                    </div>
                    <div className="visit-info-item">
                      <Calendar size={24} strokeWidth={2.5} color={getProcedureColor(visit.procedure)} />
                      <span className="visit-info-text">
                        {new Date(visit.date).toLocaleDateString('ru-RU', { 
                          day: 'numeric', 
                          month: 'long', 
                          year: 'numeric' 
                        })}
                      </span>
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <div className="empty-icon-circle">
                <Sparkles size={80} strokeWidth={1.5} />
              </div>
              <h3 className="empty-title">Нет посещений</h3>
              <p className="empty-text">Запишитесь на спа-процедуру прямо сейчас</p>
            </div>
          )}
        </motion.section>

        <ImageSlider images={[
          '/images/spa-1.jpg',
          '/images/spa-2.jpg',
          '/images/spa-3.jpg',
          '/images/spa-4.jpg'
        ]} />
      </div>
    </div>
  )
}

export default MyVisits

